#include <RFM69.h>    
#include <SPI.h>
#include "config.h"
#include <ESP8266WiFi.h>
#include <Ticker.h>
#include <time.h>
#include <simpleDSTadjust.h>
#include <Wire.h>
#include <Adafruit_RGBLCDShield.h>



// ************************** RF Transmitter Configuration ************************************
#define NETWORKID     100  // The same on all nodes that talk to each other
#define NODEID        2    // The unique identifier of this node
#define RECEIVER      1    // The recipient of packets

//Match frequency to the hardware version of the radio on your Feather
#define FREQUENCY     RF69_433MHZ
#define ENCRYPTKEY    "sampleEncryptKey" //exactly the same 16 characters/bytes on all nodes!
#define IS_RFM69HCW   true // set to 'true' if you are using an RFM69HCW module

int16_t packetnum = 0;  // packet counter, we increment per xmission
RFM69 radio = RFM69(RFM69_CS, RFM69_IRQ, IS_RFM69HCW, RFM69_IRQN);
int failed_to_send_counter = 0; //counter increments every failed transmission, resets when successful
int radio_strengh = -100;  //RSSI variable
//*********************************************************************************************


// ********************* NTC Clock with Daylight Savings Time (DST) Correction Configuration *****************************************************
// Update time from NTP server every 1 hour
#define NTP_UPDATE_INTERVAL_SEC 1*3600
// Maximum of 3 servers
#define NTP_SERVERS "us.pool.ntp.org", "pool.ntp.org", "time.nist.gov"
// Daylight Saving Time (DST) rule configuration
// Rules work for most contries that observe DST - see https://en.wikipedia.org/wiki/Daylight_saving_time_by_country for details and exceptions
// See http://www.timeanddate.com/time/zones/ for standard abbreviations and additional information
// Caution: DST rules may change in the future
//US Eastern Time Zone (New York, Boston)
#define timezone -5 // US Eastern Time Zone
/*
  struct dstRule
  {
  char abbrev[6];    //five chars max
  uint8_t week;      //First, Second, Third, Fourth, or Last week of the month
  uint8_t dow;       //day of week, 0=Sun, 1=Mon, ... 6=Sat
  uint8_t month;     //0=Jan, 1=Feb, ... 11=Dec
  uint8_t hour;      //0-23
  int offset;        //offset from Standard Time in seconds
  };
*/
struct dstRule StartRule = {"EDT", Second, Sun, Mar, 2, 3600};    // Daylight time = UTC/GMT -4 hours
struct dstRule EndRule = {"EST", First, Sun, Nov, 2, 0};       // Standard time = UTC/GMT -5 hour
// Setup simpleDSTadjust Library rules
simpleDSTadjust dstAdjusted(StartRule, EndRule);
//***************************************************************************************************************************************************


// ************************** PIN OUTPUT/INPUT Configuration **********************************
#define SERIAL_BAUD   115200

//Only for ESP8266 feather w/ RF69_433MHZ wing
#define RFM69_CS      2
#define RFM69_IRQ     15
#define RFM69_IRQN    digitalPinToInterrupt(RFM69_IRQ )
#define RFM69_RST     16
#define LED 0    //use 0 on ESP8266 - onboard LED blinks whenever transmission/reception
//*********************************************************************************************


// **************************     Display Configuration and RGB Button     **********************************
Adafruit_RGBLCDShield lcd = Adafruit_RGBLCDShield();
#define mono_display 0x4
#define green 0x5
#define red 0x6
//************************************************************************************************************



//Adafruit IO Variables
int current_button_status = 0; //button status on adafruit.io 0=closed 1=open 
long previousMillis = 0; //variable used for timing purposes
long previousMillis2 = 0; //variable used for timing purposes
long previousMillis3 = 0; //variable used for timing purposes
long previousMillis4 = 0; //variable used for timing purposes
long debouncingMillis = 0; //variable used for timing purposes
long dropped_connectionMillis = 0; //variable used for timing purposes
unsigned long currentMillis = 0; //variable used for timing purposes
long interval = 15 * 1000; //how often we send a RF transmission in ms
long interval2 = 30 * 1000; //how often we send the status of the button to adafruit IO in ms 
long interval3 = 5 * 60 * 1000; //how often we send the open time to adafruit IO in ms 
long interval4 = 5 * 65 * 1000; //how often we send the close time to adafruit IO in ms 
long debouncing_interval = 1000; //how long we wait until we can press the physical button again in ms
long dropped_connection_interval = 70 * 1000;  //if nothing received from the internet for 70 seconds, reset the wifi
boolean last_message_was_open = true;
boolean last_message_was_closed = true;
boolean last_message_was_error = false;

//Default open/close timer times. Edit these if you would like different default times. They are in 24 hour format!
int hour_ON = 6; 
int min_ON = 0;
int hour_OFF = 19;
int min_OFF = 0;

// set up the feeds
AdafruitIO_Feed *toggleSwitch = io.feed("Chicken Coup");
AdafruitIO_Feed *timer = io.feed("Chicken Coup Timer");
AdafruitIO_Feed *timer2 = io.feed("Chicken Coup Timer 2");
AdafruitIO_Feed *error = io.feed("Chicken Coup Error Message");

bool open_alarm_has_not_gone_off = true; //makes sure open timer only runs one time every 24 hours
bool close_alarm_has_not_gone_off = true;//makes sure close timer only runs one time every 24 hours
bool error_message_1_active = false; //if transmitter doesnt get an ACK more than 10 times in a row
bool error_message_2_active = false; //if the door switch doesnt match the current button state variable

//display variables
String open_close = "closed";
int previous_minute = 12;
int previous_hour = 24;

//NTC Variables
Ticker ticker1;
int32_t tick;
// flag changed in the ticker function to start NTP Update
bool readyForNtpUpdate = false;
int month, day, year, hour_24, minute, seconds;







void setup() {
  // set up the LCD's number of columns and rows:
  lcd.begin(20, 4);
  //turn on the backlight with red led
  lcd.setBacklight(red);
  delay(100);
  lcd.clear();
  lcd.setCursor(0, 1);
  lcd.print("Initializing");

  //Seting up the Serial
  Serial.begin(SERIAL_BAUD);
  Serial.setDebugOutput(false);

  initialize_radio();

  connect_to_adafruitIO();

  //Set the Default switch status and open/close timer time
  toggleSwitch->save("Closed");
  delay(3000);
  char ONtime_buf[5];
  sprintf(ONtime_buf, "%02d:%02d", hour_ON, min_ON);
  timer->save(ONtime_buf);
  delay(3000);
  char OFFtime_buf[5];
  sprintf(OFFtime_buf, "%02d:%02d\n", hour_OFF, min_OFF);
  timer2->save(OFFtime_buf);

  //Setup the NTC clock
  updateNTP(); // Init the NTP time
  printTime(0); // print initial time time now.
  tick = NTP_UPDATE_INTERVAL_SEC; // Init the NTP update countdown ticker
  ticker1.attach(1, secTicker); // Run a 1 second interval Ticker
  printTime(tick);

  //display static text and initial times
  lcd.clear();
  //turn on the backlight with red led
  lcd.setBacklight(red);
  //print current time
  print_time(24, 1, 0, 0);
  //print open time
  lcd.setCursor(0, 1);
  lcd.print("Open at: ");
  print_time(hour_ON, min_ON, 9, 1);
  //print close time
  lcd.setCursor(0, 2);
  lcd.print("Close at: ");
  print_time(hour_OFF, min_OFF, 10, 2);
  // set the cursor to column 0, line 1
  lcd.setCursor(0, 3);
  lcd.print("Currently: ");
  lcd.setCursor(11, 3);
  lcd.print(open_close);

  io.run();
  
}





void loop() {

  delay(0); //reset the ESP8266 watchdog timer
  currentMillis = millis(); //record the current number of milliseconds elapsed since start of program


  //Get current date/time from NTC Server
  if (readyForNtpUpdate) {
    readyForNtpUpdate = false;
    printTime(0);
    updateNTP();
    Serial.print("\nUpdated time from NTP Server: ");
    printTime(0);
    Serial.print("Next NTP Update: ");
    printTime(tick);
  }


  //This section acts as an alarm clock and is how the OPEN/CLOSE Timer works
  //Checks if it is time to open the chicken Coup if the current hour is the same hour as ON hour, and greater than or equal to the ON minute,
  if (hour_24 == hour_ON && minute >= min_ON && open_alarm_has_not_gone_off) {
    delay(500);
    //open the door, broadcast "open"
    Serial.println("Publishing Open to Adafruit IO because current time is greater than on time");
    toggleSwitch->save("Open");
    current_button_status == 1;
    open_alarm_has_not_gone_off = false; //it just went off
    delay(300);
  }
  //Checks if it is time to close the chicken Coup if the current hour is the same hour as OFF hour, and greater than or equal to the OFF minute,
  if (hour_24 == hour_OFF && minute >= min_OFF && close_alarm_has_not_gone_off) {
    delay(500);
    //close the door, broadcast "close"
    Serial.println("Publishing Closed to Adafruit IO because current time is greater than off time");
    toggleSwitch->save("Closed");
    current_button_status == 0;
    close_alarm_has_not_gone_off = false; //it just went off
    delay(300);
  }
  //reset the alarms so they turn on next time (24 hours later)
  if (hour_24 > hour_ON) {
    open_alarm_has_not_gone_off = true;
  }
  if (hour_24 > hour_OFF) {
    close_alarm_has_not_gone_off = true;
  }


  //Checks if physical button is pushed
  uint8_t buttons = lcd.readButtons();
  //if the button is pushed and the toggle switch status online is "open", change status to closed, then broadcast "closed" to adafruit_IO
  if (buttons && current_button_status == 1 && currentMillis - debouncingMillis > debouncing_interval) {
    Serial.println("Publishing Closed to Adafruit IO");
    toggleSwitch->save("Closed");
    current_button_status == 0;
    debouncingMillis = currentMillis;
  }
  //if the button is pushed and the toggle switch status online is "closed", change status to open, then broadcast "open" to adafruit_IO
  else if (buttons && current_button_status == 0 && currentMillis - debouncingMillis > debouncing_interval) {
    Serial.println("Publishing Open to Adafruit IO");
    toggleSwitch->save("Open");
    current_button_status == 1;
    debouncingMillis = currentMillis;
  }

  
  //Updates the NTC clock on the LCD screen
  //this will run every time the hour or minute changes
  if ((previous_minute != minute || previous_hour != hour_24) && error_message_1_active == false && error_message_2_active == false) {
    //time to update
    LCD_clear_time();
    print_time(hour_24, minute, 0, 0);
    previous_minute = minute;
    previous_hour = hour_24;
  }


  //Constantly send data to adafruit IO because sometimes Adafruit IO gets out of sync with this sketch
  if (currentMillis - previousMillis2 > interval2) { //every 30 seconds send the status of the button
    if (current_button_status == 1) {
      toggleSwitch->save("Open");
    }
    else {
      toggleSwitch->save("Closed");
    }
    previousMillis2 = currentMillis;
  }
  if (currentMillis - previousMillis3 > interval3) { //every 5 minutes send the open timer time
    char ONtime_buf[5];
    sprintf(ONtime_buf, "%02d:%02d", hour_ON, min_ON);
    timer->save(ONtime_buf);
    previousMillis3 = currentMillis;
  }
  if (currentMillis - previousMillis4 > interval4) { //every 5 mins 25 seconds send the close timer time
    char OFFtime_buf[5];
    sprintf(OFFtime_buf, "%02d:%02d\n", hour_OFF, min_OFF);
    timer2->save(OFFtime_buf);
    previousMillis4 = currentMillis;
  }


  //If 70 seconds have passed and we have not gotten any data from Adafruit IO, reset the WiFi and reconnect to adafruit IO
  if (currentMillis - dropped_connectionMillis > dropped_connection_interval) {
    Serial.println(io.statusText());
    dropped_connectionMillis = currentMillis;
    //Send to adafruit feed
    WiFi.disconnect();
    Serial.println("Just disconnected Wifi");
    delay(200);
    Serial.println("We need to recconect to adafruitIO");
    lcd.setCursor(17, 0);
    lcd.print("IO");
    connect_to_adafruitIO();
    delay(500);
    lcd.setCursor(17, 0);
    lcd.print("  ");
  }
  else {
    io.run();
  }
  

  delay(0); //reset the ESP8266 watchdog timer

  //***TRANSMIT***
  //Either transmit button status every 10 seconds or immediately after it changes from one state to another
  if ((current_button_status == 1 && last_message_was_open == false) || (current_button_status == 1 && currentMillis - previousMillis > interval)) {
    delay(1000);
    char radiopacket[20] = "Open";
    itoa(packetnum++, radiopacket+13, 10);
    last_message_was_open = true;
    last_message_was_closed = false;
    previousMillis = currentMillis;
    Serial.print("Sending at 433MHz: "); Serial.println(radiopacket);
    delay(10);
    if (radio.sendWithRetry(RECEIVER, radiopacket, strlen(radiopacket))) { //target node Id, message as string or byte array, message length
      failed_to_send_counter = 0;
      error_message_1_active = false;
      Serial.println("Acknowledge Received");
      Blink(LED, 50, 3); //blink LED 3 times, 50ms between blinks
      radio.receiveDone(); //put radio in RX mode
    }
    else {
      Serial.println("failed");
      failed_to_send_counter += 1;
      radio.receiveDone(); //put radio in RX mode
    }
  }
  else if ((current_button_status == 0 && last_message_was_closed == false) || (current_button_status == 0 && currentMillis - previousMillis > interval)) {
    delay(1000);  // Wait 1 second between transmits, could also 'sleep' here!
    char radiopacket[20] = "Closed";
    itoa(packetnum++, radiopacket+13, 10);
    last_message_was_open = false;
    last_message_was_closed = true;
    previousMillis = currentMillis;
    delay(10);
    Serial.print("Sending at 433MHz :"); Serial.println(radiopacket);
    if (radio.sendWithRetry(RECEIVER, radiopacket, strlen(radiopacket))) { //target node Id, message as string or byte array, message length 
      failed_to_send_counter = 0;
      error_message_1_active = false;
      Serial.println("Acknowledge Received");
      Blink(LED, 50, 3); //blink LED 3 times, 50ms between blinks
      radio.receiveDone(); //put radio in RX mode
    }
    else {
      radio.receiveDone(); //put radio in RX mode
      Serial.println("failed");
      failed_to_send_counter += 1;
    }
  }
  //Checks if the TX failed to get an ACK more than 10 times
  if (failed_to_send_counter > 10 && error_message_1_active == false) {
    //Send to adafruit feed
    error->save("Error 1");
    error_message_1_active = true;
  }
  //resets the display if there are no more error messgaes
  if (error_message_1_active == false && error_message_2_active == false && last_message_was_error == true) {
    error->save("OK");
  }

  delay(0); //reset the ESP8266 watchdog timer

  
  //***RECEIVE***
  //Check if there is a message about the door switch sensor this makes sure the current button status matches the door switch sensor status
  //if it does not match, send an error message to adafruit IO
  if (radio.receiveDone()) {
    //print message received to serial
    Serial.print('['); Serial.print(radio.SENDERID); Serial.print("] ");
    Serial.print((char*)radio.DATA);
    radio_strengh=radio.RSSI;    
    if ((strstr((char *)radio.DATA, "sensorClosed")) || (strstr((char *)radio.DATA, "sensorOpen")))
    {
      //check if sender wanted an ACK
      if (radio.ACKRequested())
      {
        radio.sendACK();
        Serial.println(" - ACK sent");
      }
      Blink(LED, 40, 3); //blink LED 3 times, 40ms between blinks      
    }
    radio.receiveDone(); //put radio in RX mode
    
    if (strstr((char *)radio.DATA, "sensorOpen")) {
      //if they don't match
      if (current_button_status == 0) { //0 is close
        delay(200);
        error->save("Error 2");
        error_message_2_active = true;
        Serial.println("Just published error 2 because door switch open but current button status is closed");
      }
      else{
        error_message_2_active = false;
      }
    }
    else if (strstr((char *)radio.DATA, "sensorClosed")) {
      //if they don't match
      if (current_button_status == 1) { //1 is open
        delay(200);
        error->save("Error 2");
        error_message_2_active = true;
        Serial.println("Just published error 2 because door switch open but current button status is open");
      }
      else{
        error_message_2_active = false;
      }
    }
    //Print out the radio stregth to the LCD 
    Serial.print("printed: ");
    Serial.println(radio_strengh);
    if(error_message_1_active == false && error_message_2_active == false){
        lcd.setCursor(16, 0);
        lcd.print("    ");
        lcd.setCursor(16, 0);
        lcd.print(radio_strengh);
      }
  }
}





//----------------------- Supporting Functions -------------------------------
//Initialize RFM69
void initialize_radio() {
  //Setting up the RF Module
  // Hard Reset the RFM module
  pinMode(RFM69_RST, OUTPUT);
  digitalWrite(RFM69_RST, HIGH);
  delay(100);
  digitalWrite(RFM69_RST, LOW);
  delay(100);
  // Initialize radio
  radio.initialize(FREQUENCY,NODEID,NETWORKID);
  if (IS_RFM69HCW) {
    radio.setHighPower();    // Only for RFM69HCW & HW!
  }
  radio.setPowerLevel(31); // power output ranges from 0 (5dBm) to 31 (20dBm)
  
  radio.encrypt(ENCRYPTKEY);
  
  pinMode(LED, OUTPUT);
  Serial.print("\nTransmitting at ");
  Serial.print(FREQUENCY==RF69_433MHZ ? 433 : FREQUENCY==RF69_868MHZ ? 868 : 915);
  Serial.println(" MHz");
}

//Connect to adafruit IO
void connect_to_adafruitIO() {
  io.connect();
  // set up a message handler for the 'toggleSwitch' feed.
  // the handleMessage function (defined below)
  // will be called whenever a message is
  // received from adafruit io.
  toggleSwitch->onMessage(handleMessage);
  timer->onMessage(handletimeMessage);
  timer2->onMessage(handletime2Message);
  error->onMessage(errorMessage);
  // wait for a connection
  while (io.status() < AIO_CONNECTED) {
    Serial.print(".");
    delay(500);
  }
  //Print we are connected to Adafruit IO
  Serial.println();
  Serial.println(io.statusText());
}


//Parses a string and separates string into two strings based on a separator
//This is used when getting timer data from Adafruit IO
String getValue(String data, char separator, int index)
{
  int found = 0;
  int strIndex[] = { 0, -1 };
  int maxIndex = data.length() - 1;

  for (int i = 0; i <= maxIndex && found <= index; i++) {
    if (data.charAt(i) == separator || i == maxIndex) {
      found++;
      strIndex[0] = strIndex[1] + 1;
      strIndex[1] = (i == maxIndex) ? i + 1 : i;
    }
  }
  return found > index ? data.substring(strIndex[0], strIndex[1]) : "";
}


// This function is called whenever any 'toggleSwitch' feed message is received from Adafruit IO. 
//it was attached to the 'toggleSwitch' feed in the connect_to_adafruitIO() function above.
void handleMessage(AdafruitIO_Data *data) {
  dropped_connectionMillis = currentMillis;
  Serial.print("From Adafruit IO, received <- ");
  Serial.println(data->toString());
  if (data->toString() == "Open") {
    current_button_status = 1;
    // write the current state to the led and display
    open_close = "open";
    if (error_message_1_active == false && error_message_2_active == false) {
      lcd.setBacklight(green);
      LCD_clear_open_close();
      lcd.setCursor(11, 3);
      lcd.print(open_close);
    }
  }
  else if (data->toString() == "Closed") {
    current_button_status = 0;
    // write the current state to the led and display
    open_close = "closed";
    if (error_message_1_active == false && error_message_2_active == false) {
      lcd.setBacklight(red);
      LCD_clear_open_close();
      lcd.setCursor(11, 3);
      lcd.print(open_close);
    }
  }
  else
    Serial.println("Error with open/close message");
}


//This function is called whenever an 'timer' feed message is received from Adafruit IO. 
//it was attached to the 'timer' feed in the connect_to_adafruitIO() function above.
void handletimeMessage(AdafruitIO_Data *data) {
  dropped_connectionMillis = currentMillis;
  Serial.print("From Adafruit IO, received <- ");
  Serial.println(data->toString());

  hour_ON = getValue(data->toString(), ':', 0).toInt();
  min_ON = getValue(data->toString(), ':', 1).toInt();

  Serial.print("The ON Hour is: ");
  Serial.println(hour_ON);

  Serial.print("The ON Minute is: ");
  Serial.println(min_ON);

  if (error_message_1_active == false) {
    LCD_clear_open_time();
    print_time(hour_ON, min_ON, 9, 1);
  }
}


//This function is called whenever an 'timer2' feed message is received from Adafruit IO. 
//it was attached to the 'timer2' feed in the connect_to_adafruitIO() function above.
void handletime2Message(AdafruitIO_Data *data2) {
  dropped_connectionMillis = currentMillis;
  Serial.print("From Adafruit IO, received <- ");
  Serial.println(data2->toString());

  hour_OFF = getValue(data2->toString(), ':', 0).toInt();
  min_OFF = getValue(data2->toString(), ':', 1).toInt();

  Serial.print("The OFF Hour is: ");
  Serial.println(hour_OFF);

  Serial.print("The OFF Minute is: ");
  Serial.println(min_OFF);

  if (error_message_1_active == false) {
    LCD_clear_close_time();
    print_time(hour_OFF, min_OFF, 10, 2);
  }
}


// this function is called whenever an 'error' feed message is received from Adafruit IO. 
//it was attached to the 'error' feed in the connect_to_adafruitIO() function above.
void errorMessage(AdafruitIO_Data *data3) {
  dropped_connectionMillis = currentMillis;
  Serial.print("From Adafruit IO, received <- ");
  Serial.println(data3->toString());
  if (data3->toString() == "Error 1") {
    last_message_was_error = true;
    lcd.setBacklight(mono_display);
    lcd.clear();
    lcd.setCursor(0, 1);
    lcd.print(" ERROR! ");
    lcd.setCursor(0, 2);
    lcd.print("Try restarting the");
    lcd.setCursor(0, 3);
    lcd.print("outdoor receiver.");
    initialize_radio();
  }
  else if (data3->toString() == "Error 2") {
    last_message_was_error = true;
    lcd.setBacklight(mono_display);
    lcd.clear();
    lcd.setCursor(0, 1);
    lcd.print(" ERROR! ");
    lcd.setCursor(0, 2);
    lcd.print("Door or switch");
    lcd.setCursor(0, 3);
    lcd.print("sensor issue.");
  }
  else if (data3->toString() == "OK") {
    last_message_was_error = false;
    //display static text and initial times
    lcd.clear();
    if (current_button_status == 0) {
      lcd.setBacklight(red);
    }
    else {
      lcd.setBacklight(green);
    }
    print_time(hour_24, minute, 0, 0);
    lcd.setCursor(0, 1);
    lcd.print("Open at: ");
    print_time(hour_ON, min_ON, 9, 1);
    lcd.setCursor(0, 2);
    lcd.print("Close at: ");
    print_time(hour_OFF, min_OFF, 10, 2);
    lcd.setCursor(0, 3);
    lcd.print("Currently: ");
    lcd.setCursor(11, 3);
    lcd.print(open_close);
    lcd.setCursor(16, 0);
    lcd.print("    ");
    lcd.setCursor(16, 0);
    lcd.print(radio_strengh);
  }
}


void Blink(byte PIN, byte DELAY_MS, byte loops)
{
  for (byte i=0; i<loops; i++)
  {
    digitalWrite(PIN,HIGH);
    delay(DELAY_MS);
    digitalWrite(PIN,LOW);
    delay(DELAY_MS);
  }
}


// NTP timer ticker
void secTicker()
{
  tick--;
  if (tick <= 0)
  {
    readyForNtpUpdate = true;
    tick = NTP_UPDATE_INTERVAL_SEC; // Re-arm
  }

  printTime(0);  // Uncomment if you want to see time printed every second
}


// NTP update function
void updateNTP() {
  configTime(timezone * 3600, 0, NTP_SERVERS);
  delay(500);
  while (!time(nullptr)) {
    Serial.print("#");
    delay(1000);
  }
}


//This function updates the NTC variables
void printTime(time_t offset)
{
  char buf[30];
  char *dstAbbrev;
  time_t t = dstAdjusted.time(&dstAbbrev) + offset;
  struct tm *timeinfo = localtime (&t);
  int hour = (timeinfo->tm_hour + 11) % 12 + 1; // take care of noon and midnight
  month = timeinfo->tm_mon + 1;
  day = timeinfo->tm_mday;
  year = timeinfo->tm_year + 1900;
  hour_24 = timeinfo->tm_hour;
  minute =  timeinfo->tm_min;
  seconds = timeinfo->tm_sec;
}


//This function takes 24 hour hour input and minute input and prints the 12 hour time with AM/PM on an LCD.
//it prints time starting at column, row.
void print_time(int hour_24_input, int min_input, int column, int row) {
  //calcuate 12 hour time
  int hour_12 = (hour_24_input + 11) % 12 + 1; // take care of noon and midnight
  char am_pm;
  if (hour_24_input >= 12 && hour_24_input < 24)
    am_pm = 'p';
  else
    am_pm = 'a';

  char min_buf[2];
  sprintf(min_buf, "%02d" , min_input);

  //print it out
  if (hour_12 < 9) {
    lcd.setCursor(column, row); //column, row
    lcd.print(hour_12);
    lcd.setCursor(column + 1, row); //column, row
    lcd.print(':');
    lcd.setCursor(column + 2, row); //column, row
    lcd.print(min_buf);
    lcd.setCursor(column + 5, row); //column, row
    lcd.print(am_pm);
    lcd.setCursor(column + 6, row); //column, row
    lcd.print('m');
  }
  else {
    lcd.setCursor(column, row); //column, row
    lcd.print(hour_12);
    lcd.setCursor(column + 2, row); //column, row
    lcd.print(':');
    lcd.setCursor(column + 3, row); //column, row
    lcd.print(min_buf);
    lcd.setCursor(column + 6, row); //column, row
    lcd.print(am_pm);
    lcd.setCursor(column + 7, row); //column, row
    lcd.print('m');
  }
  lcd.setCursor(16, 0);
  lcd.print("    ");
  lcd.setCursor(16, 0);
  lcd.print(radio_strengh);
}


//Function used to erase previous text
void LCD_clear_time() {
  lcd.setCursor(0, 0); //column, row
  lcd.print("                  ");
}


//Function used to erase previous text
void LCD_clear_open_time() {
  lcd.setCursor(8, 1); //column, row
  lcd.print("            ");
}


//Function used to erase previous text
void LCD_clear_close_time() {
  lcd.setCursor(9, 2); //column, row
  lcd.print("           ");
}


//Function used to erase previous text
void LCD_clear_open_close() {
  lcd.setCursor(10, 3); //column, row
  lcd.print("        ");
}

